import { Router } from 'express';
import { consultationController } from './controller';
import { authenticateUser, requireVendor } from '@/middlewares/auth';
import { checkFeature } from '@/middlewares/featureGate';
import { vendorService } from '@/modules/vendors/service';

const router = Router();

// Attach vendor to req for downstream use
async function attachVendor(req: any, res: any, next: any) {
  if (req.user) {
    const { data: vendor } = await vendorService.getVendorByUserId(req.user.id);
    req.vendor = vendor || null;
  }
  next();
}

// Vendor: create consultation form
router.post('/forms', 
  authenticateUser, 
  requireVendor, 
  attachVendor, 
  checkFeature('consultation_forms'), 
  consultationController.createForm
);

// Vendor: list forms
router.get('/forms', 
  authenticateUser, 
  requireVendor, 
  attachVendor, 
  checkFeature('consultation_forms'), 
  consultationController.listForms
);

// Public: get form with questions (for customers to fill)
router.get('/forms/:id', consultationController.getForm);

// Public: submit form response
router.post('/forms/:form_id/respond', consultationController.submitResponse);

// Vendor: get form responses
router.get('/forms/:id/responses', 
  authenticateUser, 
  requireVendor, 
  attachVendor, 
  checkFeature('consultation_forms'), 
  consultationController.getResponses
);

// Vendor: get form analytics
router.get('/forms/:id/analytics', 
  authenticateUser, 
  requireVendor, 
  attachVendor, 
  checkFeature('consultation_forms'), 
  consultationController.getAnalytics
);

// Vendor: duplicate form
router.post('/forms/:id/duplicate', 
  authenticateUser, 
  requireVendor, 
  attachVendor, 
  checkFeature('consultation_forms'), 
  consultationController.duplicateForm
);

// Vendor: update form status
router.patch('/forms/:id/status', 
  authenticateUser, 
  requireVendor, 
  attachVendor, 
  checkFeature('consultation_forms'), 
  consultationController.updateFormStatus
);

export default router; 